# Pattern B Deployment Guide  
## Running Trained YOLOv8n ONNX on Raspberry OS

This guide documents the **correct and reliable method** for deploying a trained
YOLOv8n ONNX model on Raspberry OS using **Pattern B**:

> **Pattern B**  
> - Python virtual environment remains on the Raspberry Pi system disk  
> - Trained AI artifacts (ONNX, names, recordings) reside on an external drive  

This approach is fully compatible with the **TrainAI YOLOv8n → ONNX guide** and is
recommended for long-running, stable deployments.

---

## Why Pattern B is recommended

Python virtual environments contain:
- Absolute paths
- Interpreter bindings
- Activation scripts tied to their creation location

Moving a virtual environment directory frequently results in:
- Broken activation
- Import errors
- Subtle runtime failures

Pattern B avoids these issues by keeping:
- **Infrastructure** (Python + venv) on the Pi
- **Data and models** on removable or large external storage

This separation improves reliability and simplifies recovery.

---

## Target directory layout

```
Raspberry Pi SD card:
└── /home/pi/
    ├── aicam-venv/
    └── (system files)

External drive:
└── /media/pi/ai/
    └── projects/
        └── aicam_drone/
            ├── AICAM_Drone_YOLOv8n.py
            ├── models/
            │   ├── drone_yolov8n.onnx
            │   └── drone.names
            └── recordings/
```

---

## 1. Mount the external drive

Create a mount point:
```bash
sudo mkdir -p /media/pi/ai
```

Identify the drive:
```bash
lsblk -f
```

Mount the device (example):
```bash
sudo mount /dev/sda1 /media/pi/ai
```

### Persistent mounting

Retrieve the UUID:
```bash
sudo blkid
```

Edit `/etc/fstab`:
```bash
sudo nano /etc/fstab
```

Example entry:
```
UUID=XXXX-XXXX  /media/pi/ai  ext4  defaults,noatime  0  2
```

Reboot once to confirm correct mounting.

---

## 2. Create project directories on the external drive

```bash
mkdir -p /media/pi/ai/projects/aicam_drone/models
mkdir -p /media/pi/ai/projects/aicam_drone/recordings
```

---

## 3. Copy trained artifacts

From the system used during the TrainAI workflow:

```bash
rsync -a   AICAM_Drone_YOLOv8n.py   models/drone_yolov8n.onnx   models/drone.names   /media/pi/ai/projects/aicam_drone/
```

Only trained artifacts and runtime code belong on the external drive.

---

## 4. Create the Python virtual environment on the Pi

Create the virtual environment on the SD card:

```bash
cd /home/pi
python3 -m venv aicam-venv
source aicam-venv/bin/activate
python -m pip install --upgrade pip
```

Install dependencies consistent with the TrainAI guide:

```bash
python -m pip install ultralytics onnx opencv-python-headless
```

This environment can be reused across reboots and updates.

---

## 5. Freeze dependencies (recommended)

```bash
python -m pip freeze > /home/pi/aicam_requirements.txt
```

This file enables fast recovery or rebuild if needed.

---

## 6. Run the detector using Pattern B

Activate the virtual environment:
```bash
source /home/pi/aicam-venv/bin/activate
```

Run the application from the external drive:
```bash
python3 /media/pi/ai/projects/aicam_drone/AICAM_Drone_YOLOv8n.py   --yolo-model /media/pi/ai/projects/aicam_drone/models/drone_yolov8n.onnx   --yolo-names /media/pi/ai/projects/aicam_drone/models/drone.names   --yolo-imgsz 640   --conf 0.6
```

Ensure that `--yolo-imgsz` matches the ONNX export size.

---

## 7. Permissions adjustment

If file access issues occur:
```bash
sudo chown -R pi:pi /media/pi/ai/projects/aicam_drone
```

---

## Common issues and causes

| Issue | Cause |
|------|------|
| Import errors | Virtual environment not activated |
| No detections | Class name mismatch |
| Invalid boxes | Image size mismatch |
| ONNX load failure | Incorrect export options |
| Slow startup | venv placed on external drive |

---

## Deployment model

- Training and ONNX export occur once
- ONNX models are immutable artifacts
- Python virtual environment is system infrastructure
- External drive stores data and models only
- Post-processing (NMS) occurs in `AICAM_Drone_YOLOv8n.py`

---

## Pattern B checklist

- [ ] Virtual environment located in `/home/pi`
- [ ] External drive mounted before execution
- [ ] ONNX exported with `nms=False`
- [ ] `drone.names` matches class list
- [ ] Image size consistent across training and runtime

---

End of Pattern B deployment guide.
